#include <cl/client_api.h>

#include <stdio.h>

#define TEXT_BOLD(text) "\033[1m" text "\033[0m"
#define TEXT_GREEN(text) "\033[32m" text "\033[0m"
#define TEXT_RED(text) "\033[31m" text "\033[0m"

const char* state_to_string(cl_eth_link_state_t state) {
    switch (state) {
        case CL_ETH_LINK_STATE_DOWN: return TEXT_RED("Down");
        case CL_ETH_LINK_STATE_UP: return TEXT_GREEN("Up");
        default: return "Unknown";
    }
}

void mode_to_string(char* buffer, size_t buffer_size, cl_eth_link_mode_t mode) {

    char* ptr = buffer;
    if (mode & CL_ETH_LINK_MODE_10MBIT) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "10MBIT | ");
    }
    if (mode & CL_ETH_LINK_MODE_100MBIT) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "100MBIT | ");
    }

    if (mode & CL_ETH_LINK_MODE_1000MBIT) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "1000MBIT | ");
    }

    if (mode & CL_ETH_LINK_MODE_DUPLEX_HALF) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "DUPLEX_HALF | ");
    }

    if (mode & CL_ETH_LINK_MODE_DUPLEX_FULL) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "DUPLEX_FULL | ");
    }

    if (mode & CL_ETH_LINK_MODE_FOLLOWER) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "FOLLOWER | ");
    }

    if (mode & CL_ETH_LINK_MODE_LEADER) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "LEADER | ");
    }

    if (mode & CL_ETH_LINK_MODE_ANEG) {
        ptr += snprintf(ptr, buffer_size - (ptr-buffer), "ANEG | ");
    }

    if ('\0' == buffer[0]) {
        snprintf(buffer, buffer_size, "<NOTHING SET>");
    } else {
        *(ptr -3) = '\0';
    }
}

int main(int argc, char** argv)
{
    // Unused
    (void) argc;
    (void) argc;

    if (CL_STATUS_SUCCESS != cl_client_init())
    {
        perror("Failed to initialize client api, aborting");
        return -1;
    }

    uint16_t num_devices;

    puts("Devices:");

    if (CL_STATUS_SUCCESS == cl_scan_for_devices(&num_devices)) {
        for (uint16_t id = 0; id < num_devices; ++id) {

            cl_identify_device(id);

            char device_name[128];
            cl_get_device_name(id, device_name, sizeof(device_name));

            uint16_t numInterfaces;
            cl_get_num_interfaces(id, &numInterfaces);

            printf("  Device: %s\n", device_name);

            for (uint16_t interface = 0; interface < numInterfaces; ++interface) {
                char interface_name[128];
                cl_get_interface_name(id, interface, interface_name, sizeof(interface_name));

                cl_identify_interface(id, interface);

                printf("    Interface: %s\n", interface_name);

                cl_interface_status_t interfaceStatus;
                cl_get_interface_status(id, interface, &interfaceStatus);

                switch (interfaceStatus.type) {
                case CL_INTERFACE_T1:
                case CL_INTERFACE_T:
                {
                    const char* link_state = state_to_string(interfaceStatus.phy_interface.eth.link_state);
                    char config_buffer[128];
                    mode_to_string(config_buffer, sizeof(config_buffer), interfaceStatus.phy_interface.eth.link_mode);
                    printf("      Link is " TEXT_BOLD("%s") " with configuration: %s\n", link_state, config_buffer);
                }
                    break;
                default:
                    puts("      Not an ethernet interface, ignoring");
                }
            }
        }
    } else {
        perror("Failed to scan for devices");
        return -1;
    }

    if (0 == num_devices) {
        puts("  None found");
    }

    return 0;
}
