#include <cl/client_api.h>

#include <exception>
#include <array>
#include <string>
#include <string_view>
#include <cstring>
#include <iostream>

class ClientApiException : public std::exception
{
public:
    ClientApiException(const std::string_view& what)
        : m_what(what)
    {}

    const char* what() const noexcept override {
        return m_what.c_str();
    }

private:
    std::string m_what;
};

class ClientApi {
public:
    ClientApi()
    {
        if (CL_STATUS_SUCCESS != cl_client_init())
        {
            throw ClientApiException("Failed to initialize cl_client");
        }
    }

    bool scanDevices()
    {
        return CL_STATUS_SUCCESS == cl_scan_for_devices(&m_numDevices);
    }

    template<typename Func>
    void for_each_device(Func f) {
        for (uint16_t device_id = 0; device_id < m_numDevices; ++device_id)
        {
            f(device_id);
        }
    }

    uint16_t numDevices() const noexcept
    { return m_numDevices; }

    ~ClientApi()
    {
        cl_client_shutdown();
    }

private:
    uint16_t m_numDevices;
};

#define TEXT_BOLD "\033[1m"
#define TEXT_RESET "\033[0m"

int main(int, char**)
{
    ClientApi cl_api;
    if (cl_api.scanDevices()) {
        if (cl_api.numDevices()) {
            cl_api.for_each_device([](cl_device_id_t id) {
                std::array<char, 128> deviceNameBuffer;
                cl_get_device_name(id, deviceNameBuffer.data(), deviceNameBuffer.size());
                std::string_view deviceName(deviceNameBuffer.data(), std::strlen(deviceNameBuffer.data()));

                uint16_t numInterfaces;
                cl_get_num_interfaces(id, &numInterfaces);

                std::cout << "  Device: " << deviceName << '\n';

                for (uint16_t interface = 0; interface < numInterfaces; ++interface) {
                    std::array<char, 128> interfaceNameBuffer;
                    cl_get_interface_name(id, interface, interfaceNameBuffer.data(), interfaceNameBuffer.size());
                    std::string_view interfaceName(interfaceNameBuffer.data(), std::strlen(interfaceNameBuffer.data()));

                    std::cout <<"    Interface: " << interfaceName << '\n';

                    cl_interface_status_t interfaceStatus;
                    cl_get_interface_status(id, interface, &interfaceStatus);

                    switch (interfaceStatus.type) {
                    case CL_INTERFACE_T1:
                    case CL_INTERFACE_T:
                    {
                        std::string state([&](){
                            switch (interfaceStatus.phy_interface.eth.link_state) {
                            case CL_ETH_LINK_STATE_DOWN: return "Down";
                            case CL_ETH_LINK_STATE_UP: return "Up";
                            default: return "Unknown";
                            }
                        }());
                        std::string config([&]() -> std::string {
                            const cl_eth_link_mode_t& mode = interfaceStatus.phy_interface.eth.link_mode;
                            std::string ret;
                            if (mode & CL_ETH_LINK_MODE_10MBIT) {
                                ret += "10MBIT | ";
                            }
                            if (mode & CL_ETH_LINK_MODE_100MBIT) {
                                ret += "100MBIT | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_1000MBIT) {
                                ret += "1000MBIT | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_DUPLEX_HALF) {
                                ret += "DUPLEX_HALF | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_DUPLEX_FULL) {
                                ret += "DUPLEX_FULL | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_FOLLOWER) {
                                ret += "FOLLOWER | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_LEADER) {
                                ret += "LEADER | ";
                            }

                            if (mode & CL_ETH_LINK_MODE_ANEG) {
                                ret += "ANEG | ";
                            }

                            if (ret.empty()) {
                                return "<NOTHING SET>";
                            } else {
                                return std::string(ret.begin(), ret.end() -3);
                            }
                        }());
                        std::cout << "      Link is " TEXT_BOLD << state << TEXT_RESET " with configuration: " << config << '\n';
                    }
                        break;
                    default:
                        std::cout << "      Not an ethernet interface, ignoring\n";
                    }
                }
            });
        } else {
            std::cout << "No devices found\n";
        }
    } else {
        std::cerr << "Failed to scan for devices!\n";
    }

    return 0;
}
