#ifndef CL_CLIENT_API_H
#define CL_CLIENT_API_H

#include <cl/defines.h>
#include <cl/types.h>

#ifdef __cplusplus
extern "C" {
#endif //__cplusplus

/****************** General ******************/

/**
 * @brief cl_scan_for_devices Will use DNS-SD to scan for cl devices on the local network
 * @param num_devices Returns the number of devices found
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_scan_for_devices(uint16_t *num_devices);

/****************** House-keeping ******************/

/**
 * @brief cl_client_init Initializes the client library
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_client_init(void);

/**
 * @brief cl_client_shutdown Deinitializes the client library
 */
CL_API void CL_CC cl_client_shutdown(void);

/****************** Device ******************/

/**
 * @brief cl_get_firmware_ean Gets the 64bit firmware EAN
 * @param device_id The id of the target device
 * @param ean Where to store the firmware EAN
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_firmware_ean(uint16_t device_id, uint64_t* ean);

/**
 * @brief cl_get_hardware_ean Gets the 64bit hardware EAN
 * @param device_id The id of the target device
 * @param ean Where to store the hardware EAN
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_hardware_ean(uint16_t device_id, uint64_t* ean);

/**
 * @brief cl_get_serial_number Gets the device's serial number
 * @param device_id The id of the target device
 * @param serial_number Where to store the serial number
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_serial_number(uint16_t device_id, uint64_t* serial_number);

/**
 * @brief cl_get_firmware_version_string Gets the firmware version as a semver string
 * @param device_id The id of the target device
 * @param buffer Buffer to store the version string in
 * @param buffer_size Size of the buffer
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_firmware_version(uint16_t device_id, char* buffer, size_t buffer_size);

/**
 * @brief cl_get_device_name Gets the device name. The returned string will be null terminated.
 * @param device_id The id of the target device
 * @param buffer A buffer to place the device name in
 * @param buffer_size The size of the buffer
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_device_name(uint16_t device_id, char* buffer, size_t buffer_size);

/**
 * @brief cl_set_device_name Sets the given null terminated device_name
 * @param device_id The id of the target device
 * @param device_name A null terminated string
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_set_device_name(uint16_t device_id, const char* device_name);

/**
 * @brief cl_identify_device Trigger identify device sequence (flash device LEDs)
 * @param device_id The id of the target device
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_identify_device(uint16_t device_id);

/****************** Interfaces ******************/
/**
 * @brief cl_get_num_interfaces Gets the number of interfaces that the device has (interface IDs
 *           will be 0..N-1, where N is what is returned from this function)
 * @param device_id The id of the target device
 * @param num_interfaces The number of interfaces this device has
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_num_interfaces(uint16_t device_id, uint16_t* num_interfaces);

/**
 * @brief cl_get_interface_name
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param buffer A buffer to place the interface name in
 * @param buffer_size The size of the buffer
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_interface_name(uint16_t device_id, uint16_t interface_id, char* buffer, size_t buffer_size);

/**
 * @brief cl_set_interface_name Sets the given null terminated interface_name
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param interface_name A null terminated string
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_set_interface_name(uint16_t device_id, uint16_t interface_id, const char* interface_name);

/**
 * @brief cl_get_interface_status Gets the interface status. 
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param interface_status The returned interface status, see \ref cl_interface_status_t for a deeper understanding.
 * @note when link is down does this return the active mode configuration and when link is up the active link configuration. 
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_interface_status(uint16_t device_id, uint16_t interface_id, cl_interface_status_t* interface_status);

/**
 * @brief cl_set_eth_interface_mode Sets the interface link mode for an ethernet interface.
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param mode The mode to set. See \ref cl_eth_link_mode_t,
 * @note that only modes exposed by \ref cl_get_supported_eth_interface_modes is possible
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_set_eth_interface_mode(uint16_t device_id, uint16_t interface_id, cl_eth_link_mode_t mode);

/**
 * @brief cl_get_eth_interface_mode gets the mode configuration
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param mode The returned mode configuration. See \ref cl_eth_link_mode_t
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_eth_interface_mode(uint16_t device_id, uint16_t interface_id, cl_eth_link_mode_t* mode);

/**
 * @brief cl_get_supported_eth_interface_modes gets the possible modes that can be configured
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @param mode The supported modes of interface. See \ref cl_eth_link_mode_t
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_get_supported_eth_interface_modes(uint16_t device_id, uint16_t interface_id, cl_eth_link_mode_t* mode);

/**
 * @brief cl_identify_interface Trigger identify interface sequence (flash interface LEDs)
 * @param device_id The id of the target device
 * @param interface_id The id of the target interface
 * @return status code, see \ref cl_status_t for possible codes
 */
CL_API cl_status_t CL_CC cl_identify_interface(uint16_t device_id, uint16_t interface_id);

// #define IPV4(octet1, octet2, octet3, octet4) (((uint32_t)octet1 << 24) | ((uint32_t)octet2 << 16) | ((uint32_t)octet3 << 8) | (uint32_t)octet4)

#endif // CL_CLIENT_API_H

#ifdef __cplusplus
}
#endif //__cplusplus
